import {
  Accordion,
  AccordionItem,
  Listbox,
  ListboxItem,
  Image,
  Badge,
  ScrollShadow,
} from "@heroui/react";
import { ChevronLeft } from "lucide-react";
import { FC } from "react";
import { SelectedFilters } from ".";
import { getBrands } from "@/routes/api";
import { useInfiniteData } from "@/hooks/useInfiniteData";
import { Brand as BrandType } from "@/types/ApiResponse";
import { useTranslation } from "react-i18next";

interface BrandSectionProps {
  selectedFilters: SelectedFilters;
  setSelectedFilters: React.Dispatch<React.SetStateAction<SelectedFilters>>;
}

const BrandSection: FC<BrandSectionProps> = ({
  selectedFilters,
  setSelectedFilters,
}) => {
  const { t } = useTranslation();

  const {
    data: brands,
    isLoading,
    hasMore,
    loadMore,
  } = useInfiniteData<BrandType>({
    fetcher: getBrands,
    perPage: 20,
    dataKey: "brands",
    forceFetchOnMount: true,
    passLocation: true,
  });

  return (
    <div className="w-full overflow-x-hidden">
      <Accordion
        variant="light"
        itemClasses={{
          base: "overflow-hidden !important",
          title: "text-xs",
          subtitle: "text-[10px] pl-1 text-foreground/50",
          content: "text-xs p-0",
          trigger: "h-10",
          indicator: "pr-2",
        }}
        defaultExpandedKeys={["1"]}
      >
        <AccordionItem
          key="1"
          aria-label={t("brand.accordionLabel")}
          title={t("brand.title")}
          subtitle={t("brand.subtitle")}
          indicator={({ isOpen }) => (
            <Badge
              color="primary"
              content={selectedFilters?.brands?.length || undefined}
              className={`transition-transform duration-300 ${selectedFilters?.brands?.length ? "" : "hidden"} ${
                isOpen ? "rotate-90" : "rotate-0"
              }`}
              classNames={{
                badge: "text-xs",
              }}
            >
              <ChevronLeft size={20} />
            </Badge>
          )}
        >
          <ScrollShadow
            hideScrollBar
            className="w-full text-xs rounded-md max-h-[25vh]"
            onScroll={(e: React.UIEvent<HTMLDivElement>) => {
              const target = e.currentTarget;
              if (
                target.scrollTop + target.clientHeight >=
                  target.scrollHeight - 20 &&
                hasMore
              ) {
                loadMore();
              }
            }}
          >
            {isLoading ? (
              <div className="space-y-2 p-2">
                {[...Array(5)].map((_, i) => (
                  <div key={i} className="flex items-center space-x-2">
                    <div className="w-5 h-5 rounded-md bg-default-200 animate-pulse" />
                    <div className="flex-1">
                      <div className="h-3 w-24 rounded bg-default-200 animate-pulse" />
                    </div>
                  </div>
                ))}
              </div>
            ) : brands.length > 0 ? (
              <Listbox
                aria-label={t("brand.selectionAria")}
                selectedKeys={
                  selectedFilters?.brands
                    ? new Set(selectedFilters.brands)
                    : undefined
                }
                selectionMode="multiple"
                variant="flat"
                onSelectionChange={(keys) => {
                  const selectedBrandSlugs = Array.from(keys as Set<string>);
                  setSelectedFilters((prev) => ({
                    ...prev,
                    brands: selectedBrandSlugs,
                  }));
                }}
              >
                {brands.map((brand) => (
                  <ListboxItem
                    key={brand.slug}
                    classNames={{ title: "text-xs", base: "px-1" }}
                    startContent={
                      <Image
                        loading="lazy"
                        src={brand.logo}
                        width={20}
                        height={20}
                        alt={brand.title}
                        className="object-cover"
                      />
                    }
                  >
                    {brand.title}
                  </ListboxItem>
                ))}
              </Listbox>
            ) : (
              <div className="text-center py-2">{t("brand.noBrands")}</div>
            )}
            {hasMore && (
              <div className="text-center py-1 text-xs text-gray-500">
                {t("brand.scrollMore")}
              </div>
            )}
          </ScrollShadow>
        </AccordionItem>
      </Accordion>
    </div>
  );
};

export default BrandSection;
