import type { NextConfig } from "next";
import { readFileSync } from "fs";
import { join } from "path";
import withPWAInit from "@ducanh2912/next-pwa";

const withPWA = withPWAInit({
  dest: "public",
  disable: process.env.NODE_ENV === "development",
  register: true,
  cacheOnFrontEndNav: true,
  aggressiveFrontEndNavCaching: true,
  reloadOnOnline: true,
});

// Read package.json to get version
const packageJson = JSON.parse(
  readFileSync(join(process.cwd(), "package.json"), "utf8")
);
const { version = "0" } = packageJson;
const isExport = process.env.NEXT_PUBLIC_SSR !== "true";

const nextConfig: NextConfig = {
  transpilePackages: ["@heroui/system", "@heroui/react"],
  turbopack: {},
  reactStrictMode: true,
  output: isExport ? "export" : undefined,
  trailingSlash: true,
  images: {
    unoptimized: isExport,
    formats: ["image/avif", "image/webp"],
    deviceSizes: [640, 750, 828, 1080, 1200, 1920, 2048, 3840],
    imageSizes: [16, 32, 48, 64, 96, 128, 256, 384],
    minimumCacheTTL: 60,
    dangerouslyAllowSVG: true,
    contentDispositionType: "attachment",
    remotePatterns: [
      {
        protocol: "https",
        hostname: "**",
      },
    ],
  },
  experimental: {
    scrollRestoration: true,
    optimizePackageImports: ["@heroui/react", "lucide-react", "react-icons"],
  },
  allowedDevOrigins: ["localhost", "127.0.0.1", "*.localhost"],
  env: {
    NEXT_PUBLIC_APP_VERSION: version,
  },
  compress: true,
  poweredByHeader: false,
  compiler: {
    removeConsole:
      process.env.NODE_ENV === "production"
        ? {
            exclude: ["error", "warn"],
          }
        : false,
  },
};

export default withPWA(nextConfig);
